<?php

/**************************************************************
* This file is part of Remository
* Copyright (c) 2006 Martin Brampton
* Issued as open source under GNU/GPL
* For support and other information, visit http://remository.com
* To contact Martin Brampton, write to martin@remository.com
*
* Remository started life as the psx-dude script by psx-dude@psx-dude.net
* It was enhanced by Matt Smith up to version 2.10
* Since then development has been primarily by Martin Brampton,
* with contributions from other people gratefully accepted
*/

class remositoryFile extends remositoryAbstract {
	/** @var int Type of metadata - 0 = Standard file, 1 = File awaiting approval, 2 = Metadata only awaiting approval */
	var $metatype = 0;
	/** @var string File name on disk or as blob */
	var $realname='';
	/** @var bool Is the name in the file system supplemented by the id? */
	var $realwithid = 0;
	/** @var bool Is the file in the local file system? */
	var $islocal='0';
	/** @var int Container ID */
	var $containerid=0;
	/** @var string File path if non-standard, derived from container */
	var $filepath='';
	/** @var string File size  */
	var $filesize='';
	/** @var string File extension */
	var $filetype='';
	/** @var string File Title for browser title bar */
	var $filetitle='';
	/** @var string File Subtitle */
	var $subtitle='';
	/** @var string File description */
	var $description='';
	/** @var string Short file description */
	var $smalldesc='';
	/** @var bool Is the short description automatically derived from the full description? */
	var $autoshort='';
	/** @var string License conditions for the file */
	var $license='';
	/** @var bool Does the user have to confirm the license conditions? */
	var $licenseagree='0';
	/** @var int Price in currency units with two decimal places */
	var $price=0;
	/** @var string Currency code e.g. GBP */
	var $currency='';
	/** @var int File download count */
	var $downloads=0;
	/** @var string URL to the file, if it is held elsewhere */
	var $url='';
	/** @var string Icon - not sure how this is used */
	var $icon='';
	/** @var bool Is this file published? */
	var $published=false;
	/** @var bool Is this file confined to registered users? */
	var $registered='2';
	/** @var User options 1=upload, 2=download, 3=both */
	var $userupload='3';
	/** @var bool Is this file recommended? */
	var $recommended=false;
	/** @var string Description of why recommended */
	var $recommend_text='';
	/** @var bool Is this file featured? */
	var $featured=false;
	/** @var date Start date for feature */
	var $featured_st_date='';
	/** @var date End date for feature */
	var $featured_end_date='';
	/** @var int Priority among featured files */
	var $featured_priority=0;
	/** @var int Sequencing number (calculated) */
	var $featured_seq=0;
	/** @var text Discussion of featured file */
	var $featured_text='';
	/** @var string Operating system for which file is intended */
	var $opsystem='';
	/** @var string Legal type - shareware, freeware, commercial, etc */
	var $legaltype='';
	/** @var text Requirements - what is the environment for running this file? */
	var $requirements='';
	/** @var Company name owning file */
	var $company='';
	/** @var date Release date */
	var $releasedate='';
	/** @var text Languages supported */
	var $languages='';
	/** @var string Company URL */
	var $company_URL='';
	/** @var string Translator name */
	var $translator='';
	/** @var string Version of this file */
	var $fileversion='';
	/** @var string Name of the author of the file */
	var $fileauthor='';
	/** @var string URL for web site of author of file */
	var $author_URL='';
	/** @var string Publisher ID */
	var $publish_id='';
	/** @var string Publish Date */
	var $publish_date=null;
	/** @var date The last modified date for the file */
	var $filedate=null;
	/** @var string Home page related to this file (URL) */
	var $filehomepage='';
	/** @var string Link to some kind of image referring to the file */
	var $screenurl='';
	/** @var bool Is this file in plain text? */
	var $plaintext=false;
	/** @var bool Is this file held in the database as a blob? */
	var $isblob=false;
	/** @var int Number of chunks for a file stored as blob in DB */
	var $chunkcount = 0;
	/** @var int Group of users that has access to this file */
	var $groupid=0;
	/** @var int Group of users who may edit this file */
	var $editgroup=0;
	/** @var string Information to be displayed during download */
	var $download_text = '';
	/** @var int The ID of the user who submitted this file */
	var $submittedby=0;
	/** @var date Date on which the file was submitted */
	var $submitdate='';
	/** @var string Custom field 1 */
	var $custom_1 = '';
	/** @var string Custom field 2 */
	var $custom_2 = '';
	/** @var string Custom field 3 */
	var $custom_3 = '';
	/** @var string Custom field 4 */
	var $custom_4 = 0;
	/** @var time stamp Custom field 5 */
	var $custom_5 = '';
	/** @var int original ID for resubmitted file */
	var $oldid=0;
	/** @var int Average rating of votes for this file */
	var $vote_value=0;
	/** @var int Count of votes for this file */
	var $vote_count=0;

	/**
	* File object constructor
	* @param int File ID from database or null
	*/
	function remositoryFile ( $id=0 ) {
		$repository =& remositoryRepository::getInstance();
		$this->id = $id;
		$this->fileversion = $repository->Default_Version;
	}

	function realName () {
		if ($this->islocal) return $this->realname;
		else return '';
	}

	function url () {
		if ($this->islocal) return '';
		else return $this->url;
	}

	function forceBools () {
		if ($this->published) $this->published=1;
		else $this->published=0;
		if ($this->licenseagree) $this->licenseagree=1;
		else $this->licenseagree=0;
		if ($this->autoshort) {
			$this->autoshort=1;
			$this->smalldesc='';
		} else $this->autoshort=0;
	}

	function notSQL () {
		return array ('id','vote_value', 'vote_count','submitdate');
	}

	function tableName () {
		return '#__downloads_files';
	}

	function timeStampField () {
		return 'submitdate';
	}

	function stripTagsFields () {
		return array ('smalldesc', 'keywords', 'filetitle', 'license', 'windowtitle', 'requirements', 'company',
		    'languages', 'company_URL', 'translator', 'fileversion', 'fileauthor', 'author_URL', 'filehomepage',
		    'screenurl');
	}

	function stripTags () {
		$fields = $this->stripTagsFields();
		foreach ($fields as $field) {
			$this->$field = strip_tags($this->$field);
		}
	}

	function validate ($extcheck=true) {
		$this->stripTags();
		$this->forceBools();
		$this->makeAutoshort();
		$this->checkLicenseagree();
		$interface =& remositoryInterface::getInstance();
		$interface->triggerMambots('remositoryValidateFile', array($this));
		if ($extcheck) $this->checkExtensionOK();
	}

	function getPlainText () {
		$interface =& remositoryInterface::getInstance();
		$database =& $interface->getDB();
		$sql = "SELECT filetext FROM #__downloads_text WHERE fileid=$this->id";
		$database->setQuery($sql);
		return $database->loadResult();
	}

	function addPostData ($adminside=false) {
		// Clear all tick boxes - will be sent by POST data if and only if tick is present
		$this->autoshort = 0;
		$this->licenseagree = 0;
		if ($adminside) {
			$this->published = 0;
			$this->featured = 0;
			$this->recommended = 0;
		}
		parent::addPostData();
	}

	function insertFileDB () {
		if (0 == $this->containerid) return;
		$interface =& remositoryInterface::getInstance();
		$database =& $interface->getDB();
		if ($this->plaintext OR $this->isblob) $id = null;
		else {
			if ($this->islocal) {
				$realname = $database->getEscaped($this->realname);
				$sql = "SELECT id FROM #__downloads_files WHERE realname='$realname' AND realwithid=0";
			}
			else {
				$url = $database->getEscaped($this->url);
				$sql = "SELECT id FROM #__downloads_files WHERE url='$url'";
			}
			$database->setQuery($sql);
			$id = $database->loadResult();
		}
		if ($id != null) {
			$this->id = $id;
			$this->updateObjectDB();
		}
		else {
			remositoryRepository::doSQL($this->insertSQL());
			$this->id = $database->insertid();
			if ($this->published) $this->incrementCounts('+1');
		}
	}

	function saveFile () {
		$default_icon = array (
		'txt' => 'document.gif',
		'exe' => 'executable.gif',
		'tar' => 'archive_tar.gif',
		'gz' => 'archive_gz.gif',
		'rar' => 'archive_rar.gif',
		'zip' => 'archive_zip.gif',
		'png' => 'pics.gif',
		'gif' => 'pics.gif',
		'jpg' => 'pics.gif',
		'pdf' => 'pdf1.gif',
		'doc' => 'word.gif',
		'rtf' => 'word.gif',
		'xls' => 'excel.gif'
		);
		if ($this->islocal) {
			$this->filetype = $this->lastPart($this->realname, '.');
			if (!$this->icon) {
				if (isset($default_icon [$this->filetype])) $this->icon = $default_icon [$this->filetype];
				else $this->icon = 'stuff1.gif';
			}
		}
		$this->metatype = 0;
		$this->oldid = 0;
		if ($this->id == 0) $this->insertFileDB();
		else $this->updateObjectDB();
	}

	function &obtainPhysical () {
		$physical =& new remositoryPhysicalFile();
		$physical->setData($this->filepath.$this->realname, $this->id, $this->isblob, $this->plaintext, $this->realwithid);
		return $physical;
	}

	function storePhysicalFile (&$physical, $extensiontitle=true, $checkExt=true) {
		$this->url = '';
	    $this->islocal = '1';
		if ($this->filetitle == '') {
			if ($extensiontitle) $this->filetitle = $physical->proper_name;
			else $this->filetitle = remositoryRepository::allButLast($physical->proper_name, '.');
		}
		if ($this->filepath) {
			if ($this->onDiskCheckFail($physical, $checkExt)) {
				echo "<script> alert('"._ERR6."'); window.history.go(-1); </script>\n";
				exit;
			}
		}
		else {
			if (!$this->plaintext) $this->isblob = 1;
			$this->getPhysicalData($physical);
		}
		$this->realwithid = 1;
		$this->saveFile();
		$newphysical =& $this->obtainPhysical();
		return $physical->moveTo($newphysical->file_path, $this->id, $newphysical->isblob, $newphysical->plaintext, true);
	}

	function downloadURL ($autodown) {
	    if ($autodown) $function = 'download';
		else $function = 'showdown';
		$repository =& remositoryRepository::getInstance();
		$downURL = $repository->RemositoryBasicFunctionURL($function,$this->id);
		$downURL = "'".str_replace('&amp;','&',$downURL)."'";
		return $downURL;
	}

	function basicDownloadLink ($autodown) {
	    if ($autodown) $function = 'download';
		else $function = 'startdown';
		if ($this->islocal AND $autodown) $fname = $this->realname;
		else $fname = null;
		$repository =& remositoryRepository::getInstance();
		$downlink = $repository->RemositoryBasicFunctionURL($function,$this->id, null, null, null, $fname);
		return $downlink;
	}

	function downloadLink ($autodown) {
		$downURL = $this->downloadURL ($autodown);
		if ($this->islocal) $addon = ' rel="nofollow">';
		elseif ($autodown == 2) $addon = ' target="_blank" rel="nofollow">';
		else $addon = ' onclick="download('.$downURL.')" rel="nofollow" target="_blank">';
		$downlink = $this->basicDownloadLink($autodown);
		$downlink = '<a href="'.$downlink.'"'.$addon;
		return $downlink;
	}

	function cloneFile () {
		$this->id = 0;
		$this->insertFileDB();
	}

	function deleteFileDB () {
		$sql = "DELETE FROM #__downloads_files WHERE id=$this->id";
		remositoryRepository::doSQL($sql);
		remositoryComment::deleteComments($this->id);
		remositoryLogEntry::deleteEntries($this->id);
		if ($this->published) $this->incrementCounts('-1');
	}

	function nameWithID () {
		if ($this->realwithid) {
			$elements = explode ('.', $this->realname);
			if (1 < count($elements)) $extension = array_pop($elements);
			else $extension = '';
			array_push ($elements, (string) $this->id);
			if ($extension) array_push ($elements, $extension);
			return implode('.', $elements);
		}
		else return $this->realname;
	}

	function filePath () {
		if ($this->filepath) return $this->filepath.$this->nameWithID();
		else return '';
	}

	function deleteFile () {
		$physical =& $this->obtainPhysical();
		$physical->delete();
		$this->deleteFileDB();
	}

	function setMetaData () {
		$interface =& remositoryInterface::getInstance();
		$interface->prependMetaTag('description', $this->smalldesc);
		if ($this->keywords) $interface->prependMetaTag('keywords', $this->keywords);
		else $interface->prependMetaTag('keywords', $this->filetitle);
	}

	function checkLicenseagree () {
		if ($this->licenseagree AND $this->license != '') $this->licenseagree = 1;
		else $this->licenseagree = 0;
	}

	function getValues (&$user) {
		$sql = "SELECT f.*, AVG(l.value) AS vote_value, COUNT(l.value) AS vote_count FROM #__downloads_files AS f LEFT JOIN #__downloads_log AS l ON l.type=3 AND l.fileid=f.id WHERE f.id = $this->id";
		if (!$user->isAdmin()) $sql .= " AND published=1";
		$sql .= ' GROUP BY f.id';
		$this->readDataBase($sql);
	}

	function evaluateVote () {
		return round($this->vote_value);
	}

	function addVote (&$user, $vote) {
		$newvote =& new remositoryLogEntry(_REM_VOTE_USER_GENERAL,$user->id,$this->id,$vote);
		$newvote->insertEntry();
		$totalvalue = $this->vote_value * $this->vote_count + $vote;
		$this->vote_count++;
		$this->vote_value = $totalvalue/$this->vote_count;
	}

	function userVoted (&$user) {
		$interface =& remositoryInterface::getInstance();
		$database =& $interface->getDB();
		$sql = "SELECT COUNT(id) FROM #__downloads_log WHERE type=3 AND userid=$user->id AND fileid=$this->id";
		remositoryRepository::doSQL($sql);
		return $database->loadResult();
	}

	function userCommented (&$user) {

		$interface =& remositoryInterface::getInstance();
		$database =& $interface->getDB();

		$sql = "SELECT count(id) FROM #__downloads_reviews WHERE itemid = $this->id AND userid = $user->id";
		$database->setQuery ($sql);
		if ($database->loadResult() == 0) return false;
		echo '<h4>'._DOWN_ALREADY_COMM.'</h4>';
		return true;
	}

	function getComments () {
		return remositoryComment::getComments($this);
	}

	function &getContainer () {
		$manager =& remositoryContainerManager::getInstance();
		$container =& $manager->getContainer(abs($this->containerid));
		return $container;
	}

	function memoContainer (&$container) {
		$this->registered = $container->registered;
		$this->userupload = $container->userupload;
		$this->groupid = $container->groupid;
		if ($this->plaintext = $container->plaintext) {
			$this->filepath = '';
			$this->isblob = 0;
		}
		else {
			if ($this->filepath = $container->filepath) $this->isblob = 0;
			else $this->isblob = 1;
		}
		$this->editgroup = $container->editgroup;
		if (!$this->filepath And !$this->plaintext) $this->isblob = 1;;
	}

	function getCategoryName () {
    	$parent = $this->getContainer();
    	return $parent->getCategoryName(true);
    }

    function getFamilyNames () {
    	$parent = $this->getContainer();
    	return $parent->getFamilyNames(true);
    }

	function incrementCounts ($by) {
		$container = $this->getContainer();
		while ($container != null) {
			$container->increment($by);
			$container=$container->getParent();
		}
	}

	function checkCountStats (&$user, &$message) {
		$repository =& remositoryRepository::getInstance();
		$maxdown = $user->isLogged() ? $repository->Max_Down_Reg_Day : $repository->Max_Down_Per_Day;
		if ($repository->Max_Down_File_Day > 0 AND $user->downloadCount($this->id) >= $repository->Max_Down_File_Day) {
			$message = '<br/>&nbsp;<br/> '._DOWN_COUNT_EXCEEDED_FILE;
			//.' '.remositoryGroup::getName($this->groupid);
			return true;
		}
		if ($maxdown > 0 AND $maxdown <= $user->totalDown()) {
			$message = '<br/>&nbsp;<br/> '._DOWN_COUNT_EXCEEDED;
			//.' '.remositoryGroup::getName($this->groupid);
			return true;
		}
		return false;
	}

	function downloadForbidden (&$user, &$message) {
	    $message = '';
		if ($user->isAdmin()) return false;
		$authoriser =& aliroAuthoriser::getInstance();
		if ($authoriser->checkPermission ('aUser', $user->id, 'download', 'remosFolder', $this->containerid)
		OR $authoriser->checkPermission ('aUser', $user->id, 'edit', 'remosFolder', $this->containerid)) {
			return $this->checkCountStats ($user, $message);
		}
		if ($user->isLogged()) {
			$container = $this->getContainer();
			return $this->forbidHandling($message, 'remositoryRegisteredRefused', '<br/>&nbsp;<br/> '._DOWN_MEMBER_ONLY_WARN.$container->name);
		}
		return $this->forbidHandling($message, 'remositoryVisitorRefused', '<br/>&nbsp;<br/> '._DOWN_REG_ONLY_WARN);
	}
	
	function forbidHandling (&$message, $event, $default) {
		$interface =& remositoryInterface::getInstance();
		$result = $interface->triggerMambots($event, array($this));
		if (empty($result) OR true === $result[0]) {
			$message = $default;
			return true;
		}
		if (is_string($result[0])) {
			$message = $result[0];
			return true;
		}
		return false;
	}

	function updatePermitted (&$user) {
		return $this->checkUpOrDel ($user, 'Allow_User_Edit');
	}

	function deletePermitted (&$user) {
		return $this->checkUpOrDel ($user, 'Allow_User_Delete');
	}
	
	function checkUpOrDel (&$user, $config) {
		if ($user->isAdmin()) return true;
		$repository =& remositoryRepository::getInstance();
		if (!$repository->$config) return false;
		if ($this->submittedby == $user->id) return true;
		$authoriser =& aliroAuthoriser::getInstance();
		return $authoriser->checkPermission ('aUser', $user->id, 'edit', 'remosFolder', $this->containerid);
	}

	function getExtension () {
		if ($this->islocal) return $this->lastPart($this->realname, '.');
		else return $this->lastPart($this->url, '.');
	}

	function checkExtensionOK () {
		if ($this->islocal) {
			if ($this->isblob OR $this->plaintext) return;
			$repository =& remositoryRepository::getInstance();
			if ($repository->badExtension ($this->realname)) exit();
			else $this->filetype = $this->getExtension();
		}
	}

	function makeAutoShort () {
		if ($this->autoshort) {
			$this->autoshort = 1;
			$repository =& remositoryRepository::getInstance();
			$max = $repository->Small_Text_Len-3;
			$plain = strip_tags($this->description);
			$plain = str_replace('&nbsp;', ' ', $plain);
			if (strlen($plain) > $max) $this->smalldesc=substr($plain,0,$max).'...';
			else $this->smalldesc = $plain;
		}
		else $this->autoshort = 0;
	}

	function &getEditSelectList ($type, $parm, &$user) {
		$repository =& remositoryRepository::getInstance();
		$clist = $repository->getSelectList(false, $this->containerid, $type, $parm, $user);
		return $clist;
	}

	function getPhysicalData (&$physicalFile) {
		$this->realname = $physicalFile->proper_name;
		$this->filedate = $physicalFile->date;
		$this->filesize = $physicalFile->size;
		if (!$this->filetitle) $this->filetitle = $physicalFile->proper_name;
		$this->islocal = 1;
		$this->url = '';
	}

	function onDiskCheckFail (&$physicalFile, $checkExt=true) {
		$repository =& remositoryRepository::getInstance();
		$physicalFile->antiLeech();
		$this->getPhysicalData($physicalFile);
		$file_path = $this->filepath.$this->realname;
   		if (file_exists($file_path) AND !$repository->Allow_Up_Overwrite) return true;
		if ($checkExt) $this->checkExtensionOK();
		return false;
	}

	function isFieldHTML ($field) {
		return in_array($field, array('description', 'smalldesc', 'license'));
	}

	function fieldSizeLimit ($field) {
		$repository =& remositoryRepository::getInstance();
		$large = array ('description', 'license');
		if (in_array($field,$large)) return $repository->Large_Text_Len;
		else return $repository->Small_Text_Len;
	}

	function getIcons () {
		return remositoryRepository::getIcons ('file_icons');
	}

	function togglePublished (&$idlist, $value) {
		$cids = implode( ',', $idlist );
		$sql = "UPDATE #__downloads_files SET published=$value". "\nWHERE id IN ($cids)";
		remositoryRepository::doSQL($sql);
	}
	
	function addSubmitterEmail () {
		$interface =& remositoryInterface::getInstance();
		$database = $interface->getDB();
		$database->setQuery("SELECT email FROM #__users WHERE id = $this->submittedby");
		$email = $database->loadResult();
		if ($email) $this->submit_email = $email;
	}

	// Alternative to use the CMS pathway instead of a separate Remository one
	function showCMSPathway () {
		$parent =& $this->getContainer();
		if (!is_null($parent)) $parent->showCMSPathway();
		$interface =& remositoryInterface::getInstance();
		$link = remositoryRepository::RemositoryRawFunctionURL('fileinfo', $this->id);
		$interface->appendPathWay($this->filetitle, $link);
	}

	function resetDownloadCounts () {
		remositoryRepository::doSQL('UPDATE #__downloads_files SET downloads=0');
	}

	function storeMemoFields ($container, $inherit=false) {
		$actions = "SET f.registered='$container->registered', f.userupload='$container->userupload', f.groupid='$container->groupid', f.editgroup='$container->editgroup'";
		if ($inherit) $sql = "UPDATE #__downloads_files AS f INNER JOIN #__downloads_structure AS s ON s.item=f.containerid $actions WHERE s.container=$container->id";
		else $sql = "UPDATE #__downloads_files AS f $actions WHERE containerid=$container->id";
		remositoryRepository::doSQL($sql);
	}

	function getFilesSQL ($published, $count=false, $containerid=0, $descendants=false, $orderby=_REM_DEFAULT_ORDERING, $search='', $limitstart=0, $limit=0, $submitter=0, $classify=0) {
		$sorter = array ('', ' ORDER BY id', ' ORDER BY filetitle', ' ORDER BY downloads DESC', ' ORDER BY submitdate DESC', ' ORDER BY u.username');
		if (!isset($sorter[$orderby]) OR $orderby == 0) $orderby = _REM_DEFAULT_ORDERING;
		if ($count) $results = 'count(f.id)';
		else $results = 'f.*, AVG(l.value) AS vote_value, COUNT(l.value) AS vote_count';
		if ($submitter) $results .= ', u.username';
		if ($descendants AND $containerid) {
			$sql = "SELECT $results FROM #__downloads_structure AS s INNER JOIN #__downloads_files AS f ON f.containerid=s.item";
			$where[] = "s.container = $containerid";
		}
		else {
			$sql = "SELECT $results FROM #__downloads_files AS f ";
			if ($containerid) $where[] = "f.containerid = $this->id";
			else $where[] = "f.metatype = 0";
		}
		if ($classify) {
			$sql .= " INNER JOIN #__downloads_file_classify AS fc ON f.id = fc.file_id ";
			$where[] = "fc.classify_id IN ($classify)";
		}
		if ($submitter) $where[] = "f.submittedby = $submitter";
		if (!$count) $sql .= ' LEFT JOIN #__downloads_log AS l ON l.type=3 AND l.fileid=f.id AND l.value != 0';
		if ($submitter OR (5 == $orderby)) $sql .= ' LEFT JOIN #__users AS u ON u.id=f.submittedby';
		if ($published) $where[] = 'f.published=1';
		$interface =& remositoryInterface::getInstance();
		if ($search) {
			$search = $interface->getEscaped($search);
			$where[] = "LOWER(f.filetitle) LIKE '%$search%'";
		}
		if (isset($where)) $sql .= ' WHERE '.implode(' AND ',$where);
		$repository =& remositoryRepository::getInstance();
		$user = $interface->getUser();
		$sql .= remositoryAbstract::visibilitySQL ($user, $repository->See_Files_no_download);
		if (!$count) {
			$sql .= ' GROUP BY f.id';
			$sql .= $sorter[$orderby];
		}
		if ($limit) $sql .= " LIMIT $limitstart,$limit";
		return $sql;
	}

	function popularLoggedFiles ($category, $max, $days, $user) {
		$interface =& remositoryInterface::getInstance();
		$database =& $interface->getDB();
		$sql = 'SELECT f.id, f.filetitle, f.autoshort, f.description, f.smalldesc, f.filedate, f.icon, f.containerid, c.name, COUNT( l.fileid ) AS downloads FROM #__downloads_log AS l, #__downloads_files AS f, #__downloads_containers AS c';
		if ($category) $sql .= ', #__downloads_structure AS s';
		$sql .= ' WHERE c.id = f.containerid AND f.published=1 AND l.type=1 AND l.fileid=f.id';
		$repository =& remositoryRepository::getInstance();
		$sql .= remositoryAbstract::visibilitySQL ($user, $repository->See_Files_no_download);
		if ($category) $sql .= " AND f.containerid=s.item AND s.container=$category";
		$sql .= " AND DATE_SUB(CURDATE(),INTERVAL $days DAY ) <= l.date";
		$sql .= " GROUP BY l.fileid ORDER BY downloads DESC LIMIT $max";
		$database->setQuery($sql);
		$files = $database->loadObjectList();
		if ($files) return $files;
		else return array();
	}

	function popularDownloadedFiles ($category, $max, $user) {
		$interface =& remositoryInterface::getInstance();
		$database =& $interface->getDB();
		$sql = 'SELECT f.id, f.downloads, f.filetitle, f.autoshort, f.description, f.smalldesc, f.filedate, f.icon, f.containerid, c.name from #__downloads_files AS f, #__downloads_containers AS c';
		if ($category) $sql .= ', #__downloads_structure AS s';
		$sql .= ' WHERE f.containerid = c.id AND f.published=1';
		$repository =& remositoryRepository::getInstance();
		$sql .= remositoryAbstract::visibilitySQL ($user, $repository->See_Files_no_download);
		if ($category) $sql .= " AND f.containerid=s.item AND s.container=$category";
		$sql .= " ORDER BY downloads DESC LIMIT $max";
		$database->setQuery($sql);
		$files = $database->loadObjectList();
		if ($files) return $files;
		else return array();
	}

	function newestFiles ($category, $max, $user) {
		$interface =& remositoryInterface::getInstance();
		$database =& $interface->getDB();
		$sql = 'SELECT f.id, f.filetitle, f.autoshort, f.description, f.smalldesc, f.filedate, f.icon, f.containerid, c.name from #__downloads_files AS f, #__downloads_containers AS c';
		if ($category) $sql .= ', #__downloads_structure AS s';
		$sql .= ' WHERE f.containerid = c.id AND f.published=1';
		$repository =& remositoryRepository::getInstance();
		$sql .= remositoryAbstract::visibilitySQL ($user, $repository->See_Files_no_download);
		if ($category) $sql .= " AND f.containerid=s.item AND s.container=$category";
		$sql .= " ORDER BY f.filedate DESC LIMIT $max";
		$database->setQuery($sql);
		$files = $database->loadObjectList();
		if ($files) return $files;
		else return array();
	}

	function getCountInContainer ($id, $published, $search='') {
		$interface =& remositoryInterface::getInstance();
		$database =& $interface->getDB();
		$sql = "SELECT COUNT(id) FROM #__downloads_files WHERE containerid = $id";
		if ($published) $sql .= ' AND published=1';
		if ($search) $sql .= " AND LOWER(filetitle) LIKE '%$search%'";
		$database->setQuery($sql);
		return $database->loadResult();
	}

	function searchFilesSQL($search_text, $seek_fields, &$user, $countOnly, $limitstart=0, $limit=0) {
		$results = $countOnly ? 'COUNT(id)' : 'id,containerid,filetitle,description,icon,filesize,downloads';
		$sql="SELECT $results FROM #__downloads_files AS f WHERE metatype = 0";
		foreach ($seek_fields as $field) $orcondition[] = "$field LIKE '%$search_text%'";
		if (isset($orcondition)) $sql .= ' AND ('.implode(' OR ', $orcondition).') ';
		else {
			echo '<br/>&nbsp;<br/>'._DOWN_SEARCH_ERR;
			exit;
		}
		$repository =& remositoryRepository::getInstance();
		$sql .= remositoryAbstract::visibilitySQL ($user, $repository->See_Files_no_download);
		$sql .= ' ORDER BY filetitle';
		if ($limit AND !$countOnly) $sql .= " LIMIT $limitstart,$limit";
		return $sql;
	}

}
